package com.infologsolutions.server;

import java.io.File;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.GnuParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.infologsolutions.server.socket.InfologChartSocketServerHandler;
import com.infologsolutions.server.socket.InfologCryptoSocketServerHandler;
import com.infologsolutions.server.socket.InfologSocketServer;
import com.infologsolutions.server.watchdog.FileSurveyor;

/**
 * A simple applet containing two sample charts in a JTabbedPane.
 * 
 * @author David Gilbert
 */
public class InfologServiceServer {
	private Logger log = LoggerFactory.getLogger(InfologServiceServer.class);

	/**
	 * Constructs the demo applet.
	 */
	public InfologServiceServer() {
	}

	public static void main(String[] args) throws Exception {
		CommandLineParser commandLineParser = new GnuParser();
		Options options = new Options();

		// Option watchDirectory = new Option("w", "watchDirectory", false, "");
		Option socket = new Option("s", false, "activate the chart socket mode");
		Option crypto = new Option("x", false, "activate the crypto socket mode");
		Option watchDirectory = new Option("w", false, "activate the watchDirectory mode");
		Option help = new Option("h", false, "print this Help");
		Option inputDirectory = new Option("i", true, "the name of the inputDirectory");
		inputDirectory.setArgName("path_to_directory");
		Option outputDirectory = new Option("o", true, "the name of the outputDirectory");
		outputDirectory.setArgName("path_to_directory");
		Option portOption = new Option("p", true, "port of the server");
		portOption.setArgName("port_number");
		Option delimiterOption = new Option("c", true, "TCP/IP character delimiter");
		delimiterOption.setArgName("char");

		options.addOption(help);
		options.addOption(socket);
		options.addOption(crypto);
		options.addOption(portOption);
		options.addOption(delimiterOption);
		options.addOption(watchDirectory);
		options.addOption(inputDirectory);
		options.addOption(outputDirectory);
		CommandLine commandLine = commandLineParser.parse(options, args);
		// Surveillance d'un rpertoire

		InfologServiceServer infologServiceServer = new InfologServiceServer();
		if (commandLine.hasOption("w")) {
			String sourcePathToLoad = commandLine.getOptionValue("i");
			String destinationPathToLoad = commandLine.getOptionValue("o");
			if (isBlank(sourcePathToLoad)) {
				printHelp(options);
			} else {
				infologServiceServer.surveillanceFichiers(sourcePathToLoad, destinationPathToLoad);
			}
		} else if (commandLine.hasOption("s")) {
			int port = -1;
			try {
				port = Integer.parseInt(commandLine.getOptionValue("p"));
			} catch (NumberFormatException e) {
				e.printStackTrace();
			}
			String destinationPathToLoad = commandLine.getOptionValue("o");
			String delimiter = commandLine.getOptionValue("c");
			infologServiceServer.serveurDeGraphes(port, destinationPathToLoad, delimiter);
		} else if (commandLine.hasOption("x")) {
			int port = -1;
			try {
				port = Integer.parseInt(commandLine.getOptionValue("p"));
			} catch (NumberFormatException e) {
				e.printStackTrace();
			}
			String delimiter = commandLine.getOptionValue("c");
			infologServiceServer.serveurCrypto(port, delimiter);

		} else if (help.hasArg()) {
			printHelp(options);
		} else {
			printHelp(options);
		}
	}

	private void serveurDeGraphes(int port, String destinationPath, String delimiter) {
		File destPath = createPath(destinationPath, true);
		if (destPath == null) {
			return;
		}
		// caractre paragraphe 
		char messageDelimiter = getMessageDelimiter(delimiter);

		InfologSocketServer infologSocketServer = new InfologSocketServer();
		InfologChartSocketServerHandler socketChartHandler = new InfologChartSocketServerHandler(infologSocketServer, messageDelimiter,
				destPath);

		infologSocketServer.demarrerServeur(port, messageDelimiter, socketChartHandler);
	}

	private void serveurCrypto(int port, String delimiter) {
		// caractre paragraphe 
		char messageDelimiter = getMessageDelimiter(delimiter);

		InfologSocketServer infologSocketServer = new InfologSocketServer();
		InfologCryptoSocketServerHandler socketChartHandler = new InfologCryptoSocketServerHandler(infologSocketServer, messageDelimiter);

		infologSocketServer.demarrerServeur(port, messageDelimiter, socketChartHandler);
	}

	private char getMessageDelimiter(String delimiter) {
		// caractre paragraphe 
		char messageDelimiter = '\u00A7';
		if (delimiter != null && delimiter.length() == 1) {
			if (log.isInfoEnabled()) {
				log.info("Caractre de fin de message : " + delimiter);
			}
			messageDelimiter = delimiter.charAt(0);
		} else {
			if (log.isInfoEnabled()) {
				log.info("Caractre de fin de message par dfaut (paragraphe) : ");
			}
		}
		return messageDelimiter;
	}

	private static void printHelp(Options options) {
		new HelpFormatter().printHelp("How to use InfologServiceServer", options);
	}

	private static boolean isBlank(String value) {
		if (value == null) {
			return true;
		} else if (value.length() == 0) {
			return true;
		} else {
			return value.trim().length() == 0;
		}
	}

	private File createPath(String path, boolean forceToTempDir) {

		File destPath = null;
		if (!isBlank(path)) {

			destPath = new File(path);
			if (!destPath.isDirectory()) {
				if (log.isErrorEnabled()) {
					log.error("The path : " + path + " is not a directory");
				}
			}
		}
		if (destPath == null && forceToTempDir) {
			String alternateDestinationDir = System.getProperty("java.io.tmpdir");
			destPath = new File(alternateDestinationDir);
			if (destPath.isDirectory()) {
				if (log.isInfoEnabled()) {
					log.info("Taking java.io.tmpdir : " + alternateDestinationDir + " as path.");
				}
			} else {
				if (log.isErrorEnabled()) {
					log.error("Error : There is no path available.");
					destPath = null;
				}

			}
		}
		return destPath;
	}

	public void surveillanceFichiers(String sourcePathToLoad, String destinationPathToLoad) {
		File sourcePath = createPath(sourcePathToLoad, false);
		File destinationPath = createPath(destinationPathToLoad, false);
		if (sourcePath == null || destinationPath == null) {
			if (log.isErrorEnabled()) {
				log.error("Invalid paths check your source and destination paths.");
			}
			return;
		}
		FileSurveyor fileSurveyor = FileSurveyor.getUniqueInstance();
		fileSurveyor.demarrerSurveillanceRepertoire(sourcePath, destinationPath);
		try {
			while (fileSurveyor.isAlive()) {
				Thread.sleep(1000);
			}
		} catch (InterruptedException e) {
			e.printStackTrace();
		}

	}

}